<?php
/**
 * Authentication Class
 * Aplikasi TTD SMAN 1 Tanjungbalai
 * Developer: T. Rifaldo Sany
 */

require_once '../config/database.php';

class Auth {
    private $db;
    
    public function __construct() {
        $this->db = getDB();
    }
    
    public function login($username, $password) {
        try {
            $stmt = $this->db->prepare("SELECT * FROM users WHERE username = ? AND is_active = 1");
            $stmt->execute([$username]);
            $user = $stmt->fetch();
            
            if ($user && password_verify($password, $user['password'])) {
                $_SESSION['user_id'] = $user['id'];
                $_SESSION['username'] = $user['username'];
                $_SESSION['role'] = $user['role'];
                $_SESSION['full_name'] = $user['full_name'];
                $_SESSION['login_time'] = time();
                
                return true;
            }
            
            return false;
        } catch (PDOException $e) {
            error_log("Login error: " . $e->getMessage());
            return false;
        }
    }
    
    public function logout() {
        session_destroy();
        return true;
    }
    
    public function isLoggedIn() {
        if (!isset($_SESSION['user_id']) || !isset($_SESSION['login_time'])) {
            return false;
        }
        
        // Check session timeout
        if (time() - $_SESSION['login_time'] > SESSION_TIMEOUT) {
            $this->logout();
            return false;
        }
        
        return true;
    }
    
    public function requireLogin() {
        if (!$this->isLoggedIn()) {
            redirect('login.php');
        }
    }
    
    public function requireRole($role) {
        $this->requireLogin();
        if ($_SESSION['role'] !== $role) {
            redirect('dashboard.php');
        }
    }
    
    public function getCurrentUser() {
        if (!$this->isLoggedIn()) {
            return null;
        }
        
        try {
            $stmt = $this->db->prepare("SELECT * FROM users WHERE id = ?");
            $stmt->execute([$_SESSION['user_id']]);
            return $stmt->fetch();
        } catch (PDOException $e) {
            error_log("Get current user error: " . $e->getMessage());
            return null;
        }
    }
    
    public function updateProfile($data) {
        try {
            $sql = "UPDATE users SET 
                    full_name = ?, 
                    email = ?, 
                    nip = ?, 
                    birth_place = ?, 
                    birth_date = ?, 
                    education = ?, 
                    position = ?, 
                    phone = ?, 
                    address = ?,
                    updated_at = CURRENT_TIMESTAMP
                    WHERE id = ?";
            
            $stmt = $this->db->prepare($sql);
            return $stmt->execute([
                $data['full_name'],
                $data['email'],
                $data['nip'],
                $data['birth_place'],
                $data['birth_date'],
                $data['education'],
                $data['position'],
                $data['phone'],
                $data['address'],
                $_SESSION['user_id']
            ]);
        } catch (PDOException $e) {
            error_log("Update profile error: " . $e->getMessage());
            return false;
        }
    }
    
    public function updateProfilePhoto($filename) {
        try {
            $stmt = $this->db->prepare("UPDATE users SET profile_photo = ? WHERE id = ?");
            return $stmt->execute([$filename, $_SESSION['user_id']]);
        } catch (PDOException $e) {
            error_log("Update profile photo error: " . $e->getMessage());
            return false;
        }
    }
    
    public function changePassword($currentPassword, $newPassword) {
        try {
            $user = $this->getCurrentUser();
            if (!$user || !password_verify($currentPassword, $user['password'])) {
                return false;
            }
            
            $hashedPassword = password_hash($newPassword, PASSWORD_DEFAULT);
            $stmt = $this->db->prepare("UPDATE users SET password = ? WHERE id = ?");
            return $stmt->execute([$hashedPassword, $_SESSION['user_id']]);
        } catch (PDOException $e) {
            error_log("Change password error: " . $e->getMessage());
            return false;
        }
    }
}
?>

