<?php
/**
 * QR Code Generator Class
 * Aplikasi TTD SMAN 1 Tanjungbalai
 * Developer: T. Rifaldo Sany
 */

require_once '../config/database.php';

class QRCodeGenerator {
    private $db;
    
    public function __construct() {
        $this->db = getDB();
    }
    
    public function generateQRCode($documentId) {
        try {
            // Get document info
            $stmt = $this->db->prepare("SELECT * FROM documents WHERE id = ? AND status = 'approved'");
            $stmt->execute([$documentId]);
            $document = $stmt->fetch();
            
            if (!$document) {
                return false;
            }
            
            // Create verification URL
            $verificationUrl = APP_URL . '/verify.php?doc=' . urlencode($document['document_number']);
            
            // Generate QR Code using online service (since we can't install QR libraries)
            $qrCodeUrl = "https://api.qrserver.com/v1/create-qr-code/?size=300x300&data=" . urlencode($verificationUrl);
            
            // Download QR Code
            $qrCodeData = file_get_contents($qrCodeUrl);
            if ($qrCodeData === false) {
                return false;
            }
            
            // Create filename
            $filename = 'qr_' . $document['document_number'] . '_' . time() . '.png';
            $qrPath = QR_CODE_PATH . '/' . $filename;
            
            // Save QR Code
            if (file_put_contents($qrPath, $qrCodeData) === false) {
                return false;
            }
            
            // Add logo to QR Code
            $this->addLogoToQRCode($qrPath);
            
            // Update document with QR code path
            $stmt = $this->db->prepare("UPDATE documents SET qr_code_path = ?, verification_url = ? WHERE id = ?");
            $stmt->execute([$filename, $verificationUrl, $documentId]);
            
            return $filename;
            
        } catch (Exception $e) {
            error_log("QR Code generation error: " . $e->getMessage());
            return false;
        }
    }
    
    private function addLogoToQRCode($qrPath) {
        try {
            // Load QR Code image
            $qrImage = imagecreatefrompng($qrPath);
            if (!$qrImage) {
                return false;
            }
            
            // Load logo
            $logoPath = BASE_PATH . '/uploads/logo_pemprovsu.png';
            if (!file_exists($logoPath)) {
                return false;
            }
            
            $logo = imagecreatefrompng($logoPath);
            if (!$logo) {
                return false;
            }
            
            // Get dimensions
            $qrWidth = imagesx($qrImage);
            $qrHeight = imagesy($qrImage);
            $logoWidth = imagesx($logo);
            $logoHeight = imagesy($logo);
            
            // Calculate logo size (20% of QR code)
            $newLogoWidth = $qrWidth * 0.2;
            $newLogoHeight = $qrHeight * 0.2;
            
            // Calculate position (center)
            $logoX = ($qrWidth - $newLogoWidth) / 2;
            $logoY = ($qrHeight - $newLogoHeight) / 2;
            
            // Create white background for logo
            $white = imagecolorallocate($qrImage, 255, 255, 255);
            $padding = 5;
            imagefilledrectangle($qrImage, 
                $logoX - $padding, 
                $logoY - $padding, 
                $logoX + $newLogoWidth + $padding, 
                $logoY + $newLogoHeight + $padding, 
                $white
            );
            
            // Resize and merge logo
            imagecopyresampled($qrImage, $logo, 
                $logoX, $logoY, 0, 0, 
                $newLogoWidth, $newLogoHeight, 
                $logoWidth, $logoHeight
            );
            
            // Save the modified QR code
            imagepng($qrImage, $qrPath);
            
            // Clean up
            imagedestroy($qrImage);
            imagedestroy($logo);
            
            return true;
            
        } catch (Exception $e) {
            error_log("Logo addition error: " . $e->getMessage());
            return false;
        }
    }
    
    public function generateQRCodeForApprovedDocument($documentId) {
        // This method is called when a document is approved
        return $this->generateQRCode($documentId);
    }
}
?>

