<?php
/**
 * Approvals Page - For Kepala Sekolah
 * Aplikasi TTD SMAN 1 Tanjungbalai
 * Developer: T. Rifaldo Sany
 */

require_once '../config/database.php';
require_once '../app/auth.php';
require_once '../app/qrcode.php';

$auth = new Auth();
$auth->requireRole('kepala_sekolah');

$db = getDB();
$error = '';
$success = '';

// Handle approval/rejection
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action']) && isset($_POST['document_id'])) {
        $action = $_POST['action'];
        $documentId = $_POST['document_id'];
        $notes = sanitize($_POST['notes'] ?? '');
        
        if ($action === 'approve') {
            try {
                $db->beginTransaction();
                
                // Update document status
                $stmt = $db->prepare("UPDATE documents SET status = 'approved', approved_by = ?, approved_at = CURRENT_TIMESTAMP WHERE id = ? AND status = 'pending'");
                $stmt->execute([$_SESSION['user_id'], $documentId]);
                
                if ($stmt->rowCount() > 0) {
                    // Log the approval
                    $logStmt = $db->prepare("INSERT INTO approval_logs (document_id, user_id, action, notes) VALUES (?, ?, 'approved', ?)");
                    $logStmt->execute([$documentId, $_SESSION['user_id'], $notes ?: 'Surat disetujui oleh Kepala Sekolah']);
                    
                    // Generate QR Code automatically
                    $qrGenerator = new QRCodeGenerator();
                    $qrCodeFilename = $qrGenerator->generateQRCode($documentId);
                    
                    $db->commit();
                    $success = 'Surat berhasil disetujui dan QR Code telah dibuat!';
                } else {
                    $db->rollback();
                    $error = 'Surat tidak ditemukan atau sudah diproses!';
                }
            } catch (PDOException $e) {
                $db->rollback();
                $error = 'Terjadi kesalahan database!';
                error_log("Approval error: " . $e->getMessage());
            }
        } elseif ($action === 'reject') {
            $rejectionReason = sanitize($_POST['rejection_reason'] ?? '');
            
            if (empty($rejectionReason)) {
                $error = 'Alasan penolakan harus diisi!';
            } else {
                try {
                    $db->beginTransaction();
                    
                    // Update document status
                    $stmt = $db->prepare("UPDATE documents SET status = 'rejected', approved_by = ?, approved_at = CURRENT_TIMESTAMP, rejection_reason = ? WHERE id = ? AND status = 'pending'");
                    $stmt->execute([$_SESSION['user_id'], $rejectionReason, $documentId]);
                    
                    if ($stmt->rowCount() > 0) {
                        // Log the rejection
                        $logStmt = $db->prepare("INSERT INTO approval_logs (document_id, user_id, action, notes) VALUES (?, ?, 'rejected', ?)");
                        $logStmt->execute([$documentId, $_SESSION['user_id'], $notes ?: 'Surat ditolak oleh Kepala Sekolah']);
                        
                        $db->commit();
                        $success = 'Surat berhasil ditolak!';
                    } else {
                        $db->rollback();
                        $error = 'Surat tidak ditemukan atau sudah diproses!';
                    }
                } catch (PDOException $e) {
                    $db->rollback();
                    $error = 'Terjadi kesalahan database!';
                    error_log("Rejection error: " . $e->getMessage());
                }
            }
        }
    }
}

// Get pending documents with pagination
$page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
$limit = 10;
$offset = ($page - 1) * $limit;

$search = isset($_GET['search']) ? sanitize($_GET['search']) : '';
$status = isset($_GET['status']) ? $_GET['status'] : 'pending';

// Build query
$whereClause = "WHERE 1=1";
$params = [];

if ($search) {
    $whereClause .= " AND (d.title LIKE ? OR d.document_number LIKE ? OR d.subject LIKE ? OR u.full_name LIKE ?)";
    $searchParam = "%$search%";
    $params[] = $searchParam;
    $params[] = $searchParam;
    $params[] = $searchParam;
    $params[] = $searchParam;
}

if ($status && in_array($status, ['pending', 'approved', 'rejected'])) {
    $whereClause .= " AND d.status = ?";
    $params[] = $status;
}

// Get total count
$countStmt = $db->prepare("SELECT COUNT(*) as total FROM documents d JOIN users u ON d.created_by = u.id $whereClause");
$countStmt->execute($params);
$totalDocuments = $countStmt->fetch()['total'];
$totalPages = ceil($totalDocuments / $limit);

// Get documents
$stmt = $db->prepare("SELECT d.*, u.full_name as creator_name, u.position as creator_position 
                     FROM documents d 
                     JOIN users u ON d.created_by = u.id 
                     $whereClause 
                     ORDER BY d.created_at DESC 
                     LIMIT $limit OFFSET $offset");
$stmt->execute($params);
$documents = $stmt->fetchAll();

$title = 'Persetujuan Surat - SMAN 1 Tanjungbalai';
$showNavbar = true;
$showSidebar = true;

include '../resources/header.php';
?>

<div class="d-flex justify-content-between align-items-center mb-4">
    <div>
        <h1 class="h3 mb-0 fw-bold">Persetujuan Surat</h1>
        <p class="text-muted mb-0">Kelola persetujuan surat dari Administrator</p>
    </div>
</div>

<?php if ($error): ?>
    <div class="alert alert-danger" role="alert">
        <i class="fas fa-exclamation-triangle me-2"></i><?php echo $error; ?>
    </div>
<?php endif; ?>

<?php if ($success): ?>
    <div class="alert alert-success" role="alert">
        <i class="fas fa-check-circle me-2"></i><?php echo $success; ?>
    </div>
<?php endif; ?>

<!-- Statistics Cards -->
<div class="row mb-4">
    <?php
    $statsStmt = $db->prepare("SELECT 
        COUNT(*) as total,
        SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending,
        SUM(CASE WHEN status = 'approved' THEN 1 ELSE 0 END) as approved,
        SUM(CASE WHEN status = 'rejected' THEN 1 ELSE 0 END) as rejected
        FROM documents");
    $statsStmt->execute();
    $stats = $statsStmt->fetch();
    ?>
    
    <div class="col-lg-3 col-md-6 mb-3">
        <div class="card bg-warning text-white">
            <div class="card-body">
                <div class="d-flex justify-content-between align-items-center">
                    <div>
                        <h6 class="card-title mb-0">Menunggu</h6>
                        <h2 class="mb-0"><?php echo $stats['pending']; ?></h2>
                    </div>
                    <div class="opacity-75">
                        <i class="fas fa-clock fa-2x"></i>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <div class="col-lg-3 col-md-6 mb-3">
        <div class="card bg-success text-white">
            <div class="card-body">
                <div class="d-flex justify-content-between align-items-center">
                    <div>
                        <h6 class="card-title mb-0">Disetujui</h6>
                        <h2 class="mb-0"><?php echo $stats['approved']; ?></h2>
                    </div>
                    <div class="opacity-75">
                        <i class="fas fa-check-circle fa-2x"></i>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <div class="col-lg-3 col-md-6 mb-3">
        <div class="card bg-danger text-white">
            <div class="card-body">
                <div class="d-flex justify-content-between align-items-center">
                    <div>
                        <h6 class="card-title mb-0">Ditolak</h6>
                        <h2 class="mb-0"><?php echo $stats['rejected']; ?></h2>
                    </div>
                    <div class="opacity-75">
                        <i class="fas fa-times-circle fa-2x"></i>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <div class="col-lg-3 col-md-6 mb-3">
        <div class="card bg-primary text-white">
            <div class="card-body">
                <div class="d-flex justify-content-between align-items-center">
                    <div>
                        <h6 class="card-title mb-0">Total</h6>
                        <h2 class="mb-0"><?php echo $stats['total']; ?></h2>
                    </div>
                    <div class="opacity-75">
                        <i class="fas fa-file-alt fa-2x"></i>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Filters -->
<div class="card mb-4">
    <div class="card-body">
        <form method="GET" class="row g-3">
            <div class="col-md-4">
                <label for="search" class="form-label">Pencarian</label>
                <input type="text" class="form-control" id="search" name="search" 
                       placeholder="Cari judul, nomor, perihal, atau pembuat..." 
                       value="<?php echo htmlspecialchars($search); ?>">
            </div>
            
            <div class="col-md-3">
                <label for="status" class="form-label">Status</label>
                <select class="form-select" id="status" name="status">
                    <option value="">Semua Status</option>
                    <option value="pending" <?php echo $status === 'pending' ? 'selected' : ''; ?>>Menunggu</option>
                    <option value="approved" <?php echo $status === 'approved' ? 'selected' : ''; ?>>Disetujui</option>
                    <option value="rejected" <?php echo $status === 'rejected' ? 'selected' : ''; ?>>Ditolak</option>
                </select>
            </div>
            
            <div class="col-md-3 d-flex align-items-end">
                <button type="submit" class="btn btn-outline-primary me-2">
                    <i class="fas fa-search me-1"></i>Cari
                </button>
                <a href="approvals.php" class="btn btn-outline-secondary">
                    <i class="fas fa-undo me-1"></i>Reset
                </a>
            </div>
            
            <div class="col-md-2 d-flex align-items-end justify-content-end">
                <small class="text-muted">
                    Total: <?php echo $totalDocuments; ?> surat
                </small>
            </div>
        </form>
    </div>
</div>

<!-- Documents List -->
<div class="card">
    <div class="card-header bg-white">
        <h6 class="mb-0 fw-bold">Daftar Surat</h6>
    </div>
    <div class="card-body p-0">
        <?php if (empty($documents)): ?>
            <div class="text-center py-5">
                <i class="fas fa-inbox fa-3x text-muted mb-3"></i>
                <h6 class="text-muted">Tidak ada surat</h6>
                <p class="text-muted mb-0">Belum ada surat yang perlu ditinjau</p>
            </div>
        <?php else: ?>
            <div class="table-responsive">
                <table class="table table-hover mb-0">
                    <thead class="table-light">
                        <tr>
                            <th>No. Surat</th>
                            <th>Judul</th>
                            <th>Pembuat</th>
                            <th>Status</th>
                            <th>Tanggal</th>
                            <th>Aksi</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($documents as $doc): ?>
                            <tr>
                                <td>
                                    <span class="fw-semibold"><?php echo $doc['document_number']; ?></span>
                                </td>
                                <td>
                                    <div class="fw-semibold"><?php echo $doc['title']; ?></div>
                                    <small class="text-muted"><?php echo $doc['subject']; ?></small>
                                </td>
                                <td>
                                    <div><?php echo $doc['creator_name']; ?></div>
                                    <small class="text-muted"><?php echo $doc['creator_position']; ?></small>
                                </td>
                                <td>
                                    <span class="status-badge status-<?php echo $doc['status']; ?>">
                                        <?php 
                                        switch($doc['status']) {
                                            case 'pending': echo 'Menunggu'; break;
                                            case 'approved': echo 'Disetujui'; break;
                                            case 'rejected': echo 'Ditolak'; break;
                                        }
                                        ?>
                                    </span>
                                </td>
                                <td>
                                    <div><?php echo date('d/m/Y', strtotime($doc['created_at'])); ?></div>
                                    <small class="text-muted"><?php echo date('H:i', strtotime($doc['created_at'])); ?></small>
                                </td>
                                <td>
                                    <div class="btn-group btn-group-sm" role="group">
                                        <a href="view_document.php?id=<?php echo $doc['id']; ?>" 
                                           class="btn btn-outline-info" title="Lihat Detail">
                                            <i class="fas fa-eye"></i>
                                        </a>
                                        
                                        <a href="../uploads/documents/<?php echo $doc['file_path']; ?>" 
                                           class="btn btn-outline-primary" title="Download File" target="_blank">
                                            <i class="fas fa-download"></i>
                                        </a>
                                        
                                        <?php if ($doc['status'] === 'pending'): ?>
                                            <button type="button" class="btn btn-outline-success" 
                                                    onclick="showApprovalModal(<?php echo $doc['id']; ?>, 'approve', '<?php echo addslashes($doc['title']); ?>')" 
                                                    title="Setujui">
                                                <i class="fas fa-check"></i>
                                            </button>
                                            
                                            <button type="button" class="btn btn-outline-danger" 
                                                    onclick="showApprovalModal(<?php echo $doc['id']; ?>, 'reject', '<?php echo addslashes($doc['title']); ?>')" 
                                                    title="Tolak">
                                                <i class="fas fa-times"></i>
                                            </button>
                                        <?php endif; ?>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
            
            <!-- Pagination -->
            <?php if ($totalPages > 1): ?>
                <div class="card-footer bg-white">
                    <nav aria-label="Page navigation">
                        <ul class="pagination pagination-sm justify-content-center mb-0">
                            <?php if ($page > 1): ?>
                                <li class="page-item">
                                    <a class="page-link" href="?page=<?php echo $page - 1; ?>&search=<?php echo urlencode($search); ?>&status=<?php echo $status; ?>">
                                        <i class="fas fa-chevron-left"></i>
                                    </a>
                                </li>
                            <?php endif; ?>
                            
                            <?php for ($i = max(1, $page - 2); $i <= min($totalPages, $page + 2); $i++): ?>
                                <li class="page-item <?php echo $i === $page ? 'active' : ''; ?>">
                                    <a class="page-link" href="?page=<?php echo $i; ?>&search=<?php echo urlencode($search); ?>&status=<?php echo $status; ?>">
                                        <?php echo $i; ?>
                                    </a>
                                </li>
                            <?php endfor; ?>
                            
                            <?php if ($page < $totalPages): ?>
                                <li class="page-item">
                                    <a class="page-link" href="?page=<?php echo $page + 1; ?>&search=<?php echo urlencode($search); ?>&status=<?php echo $status; ?>">
                                        <i class="fas fa-chevron-right"></i>
                                    </a>
                                </li>
                            <?php endif; ?>
                        </ul>
                    </nav>
                </div>
            <?php endif; ?>
        <?php endif; ?>
    </div>
</div>

<!-- Approval Modal -->
<div class="modal fade" id="approvalModal" tabindex="-1" aria-labelledby="approvalModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <form method="POST">
                <div class="modal-header">
                    <h5 class="modal-title" id="approvalModalLabel">Konfirmasi Tindakan</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <input type="hidden" name="document_id" id="modalDocumentId">
                    <input type="hidden" name="action" id="modalAction">
                    
                    <div id="approveContent" style="display: none;">
                        <div class="alert alert-success">
                            <i class="fas fa-check-circle me-2"></i>
                            Anda akan menyetujui surat: <strong id="approveTitle"></strong>
                        </div>
                        
                        <div class="mb-3">
                            <label for="approveNotes" class="form-label">Catatan (Opsional)</label>
                            <textarea class="form-control" name="notes" id="approveNotes" rows="3" 
                                      placeholder="Tambahkan catatan persetujuan..."></textarea>
                        </div>
                    </div>
                    
                    <div id="rejectContent" style="display: none;">
                        <div class="alert alert-danger">
                            <i class="fas fa-times-circle me-2"></i>
                            Anda akan menolak surat: <strong id="rejectTitle"></strong>
                        </div>
                        
                        <div class="mb-3">
                            <label for="rejectionReason" class="form-label">Alasan Penolakan <span class="text-danger">*</span></label>
                            <textarea class="form-control" name="rejection_reason" id="rejectionReason" rows="3" 
                                      placeholder="Jelaskan alasan penolakan..." required></textarea>
                        </div>
                        
                        <div class="mb-3">
                            <label for="rejectNotes" class="form-label">Catatan Tambahan (Opsional)</label>
                            <textarea class="form-control" name="notes" id="rejectNotes" rows="2" 
                                      placeholder="Catatan tambahan..."></textarea>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
                    <button type="submit" class="btn" id="modalSubmitBtn">Konfirmasi</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
function showApprovalModal(documentId, action, title) {
    document.getElementById('modalDocumentId').value = documentId;
    document.getElementById('modalAction').value = action;
    
    const approveContent = document.getElementById('approveContent');
    const rejectContent = document.getElementById('rejectContent');
    const submitBtn = document.getElementById('modalSubmitBtn');
    
    if (action === 'approve') {
        approveContent.style.display = 'block';
        rejectContent.style.display = 'none';
        document.getElementById('approveTitle').textContent = title;
        submitBtn.className = 'btn btn-success';
        submitBtn.textContent = 'Setujui Surat';
        
        // Clear rejection fields
        document.getElementById('rejectionReason').value = '';
        document.getElementById('rejectNotes').value = '';
    } else {
        approveContent.style.display = 'none';
        rejectContent.style.display = 'block';
        document.getElementById('rejectTitle').textContent = title;
        submitBtn.className = 'btn btn-danger';
        submitBtn.textContent = 'Tolak Surat';
        
        // Clear approval fields
        document.getElementById('approveNotes').value = '';
    }
    
    const modal = new bootstrap.Modal(document.getElementById('approvalModal'));
    modal.show();
}
</script>

<?php include '../resources/footer.php'; ?>

