<?php
/**
 * Create Document Page
 * Aplikasi TTD SMAN 1 Tanjungbalai
 * Developer: T. Rifaldo Sany
 */

require_once '../config/database.php';
require_once '../app/auth.php';

$auth = new Auth();
$auth->requireRole('administrator');

$error = '';
$success = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $title = sanitize($_POST['title']);
    $subject = sanitize($_POST['subject']);
    $description = sanitize($_POST['description']);
    
    if (empty($title) || empty($subject)) {
        $error = 'Judul dan perihal surat harus diisi!';
    } elseif (!isset($_FILES['document_file']) || $_FILES['document_file']['error'] !== UPLOAD_ERR_OK) {
        $error = 'File surat harus diupload!';
    } else {
        $file = $_FILES['document_file'];
        $allowedTypes = ['application/pdf', 'application/msword', 'application/vnd.openxmlformats-officedocument.wordprocessingml.document'];
        
        if (!in_array($file['type'], $allowedTypes)) {
            $error = 'Format file tidak didukung! Gunakan PDF, DOC, atau DOCX.';
        } elseif ($file['size'] > MAX_FILE_SIZE) {
            $error = 'Ukuran file terlalu besar! Maksimal ' . formatFileSize(MAX_FILE_SIZE) . '.';
        } else {
            try {
                $db = getDB();
                
                // Generate document number
                $documentNumber = generateDocumentNumber();
                
                // Upload file
                $extension = pathinfo($file['name'], PATHINFO_EXTENSION);
                $filename = $documentNumber . '_' . time() . '.' . $extension;
                $uploadPath = DOCUMENT_PATH . '/' . $filename;
                
                if (move_uploaded_file($file['tmp_name'], $uploadPath)) {
                    // Save to database
                    $stmt = $db->prepare("INSERT INTO documents (document_number, title, subject, description, file_path, file_name, file_size, created_by) VALUES (?, ?, ?, ?, ?, ?, ?, ?)");
                    
                    if ($stmt->execute([$documentNumber, $title, $subject, $description, $filename, $file['name'], $file['size'], $_SESSION['user_id']])) {
                        // Log the action
                        $documentId = $db->lastInsertId();
                        $logStmt = $db->prepare("INSERT INTO approval_logs (document_id, user_id, action, notes) VALUES (?, ?, 'submitted', ?)");
                        $logStmt->execute([$documentId, $_SESSION['user_id'], "Surat dibuat dengan nomor: $documentNumber"]);
                        
                        $success = "Surat berhasil dibuat dengan nomor: $documentNumber";
                        
                        // Clear form
                        $_POST = [];
                    } else {
                        $error = 'Gagal menyimpan data surat!';
                        unlink($uploadPath); // Delete uploaded file
                    }
                } else {
                    $error = 'Gagal mengupload file!';
                }
            } catch (PDOException $e) {
                $error = 'Terjadi kesalahan database!';
                error_log("Create document error: " . $e->getMessage());
            }
        }
    }
}

$title = 'Buat Surat - SMAN 1 Tanjungbalai';
$showNavbar = true;
$showSidebar = true;

include '../resources/header.php';
?>

<div class="d-flex justify-content-between align-items-center mb-4">
    <div>
        <h1 class="h3 mb-0 fw-bold">Buat Surat Baru</h1>
        <p class="text-muted mb-0">Buat surat digital untuk diajukan ke Kepala Sekolah</p>
    </div>
    <a href="documents.php" class="btn btn-outline-secondary">
        <i class="fas fa-arrow-left me-1"></i>Kembali
    </a>
</div>

<?php if ($error): ?>
    <div class="alert alert-danger" role="alert">
        <i class="fas fa-exclamation-triangle me-2"></i><?php echo $error; ?>
    </div>
<?php endif; ?>

<?php if ($success): ?>
    <div class="alert alert-success" role="alert">
        <i class="fas fa-check-circle me-2"></i><?php echo $success; ?>
        <div class="mt-2">
            <a href="documents.php" class="btn btn-sm btn-success">Lihat Daftar Surat</a>
            <a href="create_document.php" class="btn btn-sm btn-outline-success">Buat Surat Lagi</a>
        </div>
    </div>
<?php endif; ?>

<div class="row">
    <div class="col-lg-8">
        <div class="card">
            <div class="card-header bg-white">
                <h6 class="mb-0 fw-bold">Informasi Surat</h6>
            </div>
            <div class="card-body">
                <form method="POST" enctype="multipart/form-data" class="needs-validation" novalidate>
                    <div class="mb-3">
                        <label for="title" class="form-label fw-semibold">Judul Surat <span class="text-danger">*</span></label>
                        <input type="text" class="form-control" id="title" name="title" 
                               placeholder="Masukkan judul surat" 
                               value="<?php echo $_POST['title'] ?? ''; ?>" required>
                        <div class="invalid-feedback">
                            Judul surat harus diisi!
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="subject" class="form-label fw-semibold">Perihal <span class="text-danger">*</span></label>
                        <input type="text" class="form-control" id="subject" name="subject" 
                               placeholder="Masukkan perihal surat" 
                               value="<?php echo $_POST['subject'] ?? ''; ?>" required>
                        <div class="invalid-feedback">
                            Perihal surat harus diisi!
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="description" class="form-label fw-semibold">Deskripsi</label>
                        <textarea class="form-control" id="description" name="description" rows="4" 
                                  placeholder="Masukkan deskripsi atau keterangan tambahan"><?php echo $_POST['description'] ?? ''; ?></textarea>
                        <small class="text-muted">Opsional - Tambahkan keterangan atau catatan khusus</small>
                    </div>
                    
                    <div class="mb-4">
                        <label for="document_file" class="form-label fw-semibold">File Surat <span class="text-danger">*</span></label>
                        <input type="file" class="form-control" id="document_file" name="document_file" 
                               accept=".pdf,.doc,.docx" required>
                        <div class="invalid-feedback">
                            File surat harus diupload!
                        </div>
                        <small class="text-muted">
                            Format yang didukung: PDF, DOC, DOCX. Maksimal <?php echo formatFileSize(MAX_FILE_SIZE); ?>
                        </small>
                    </div>
                    
                    <div class="d-flex gap-2">
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-paper-plane me-1"></i>Kirim Surat
                        </button>
                        <button type="reset" class="btn btn-outline-secondary">
                            <i class="fas fa-undo me-1"></i>Reset Form
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <div class="col-lg-4">
        <!-- Information Card -->
        <div class="card mb-4">
            <div class="card-header bg-primary text-white">
                <h6 class="mb-0 fw-bold">
                    <i class="fas fa-info-circle me-2"></i>Informasi Penting
                </h6>
            </div>
            <div class="card-body">
                <div class="mb-3">
                    <h6 class="fw-semibold">Proses Persetujuan</h6>
                    <ol class="small mb-0">
                        <li>Surat dibuat oleh Administrator</li>
                        <li>Surat dikirim ke Kepala Sekolah</li>
                        <li>Kepala Sekolah meninjau dan memutuskan</li>
                        <li>Status surat diperbarui</li>
                        <li>QR Code dibuat untuk surat yang disetujui</li>
                    </ol>
                </div>
                
                <div class="mb-3">
                    <h6 class="fw-semibold">Format File</h6>
                    <ul class="small mb-0">
                        <li>PDF (Recommended)</li>
                        <li>Microsoft Word (.doc, .docx)</li>
                        <li>Maksimal <?php echo formatFileSize(MAX_FILE_SIZE); ?></li>
                    </ul>
                </div>
                
                <div>
                    <h6 class="fw-semibold">Tips</h6>
                    <ul class="small mb-0">
                        <li>Pastikan file dapat dibaca dengan jelas</li>
                        <li>Gunakan format PDF untuk hasil terbaik</li>
                        <li>Periksa kembali informasi sebelum mengirim</li>
                    </ul>
                </div>
            </div>
        </div>
        
        <!-- Status Legend -->
        <div class="card">
            <div class="card-header bg-white">
                <h6 class="mb-0 fw-bold">Status Surat</h6>
            </div>
            <div class="card-body">
                <div class="d-flex align-items-center mb-2">
                    <span class="status-badge status-pending me-2">Menunggu</span>
                    <small>Menunggu persetujuan Kepala Sekolah</small>
                </div>
                <div class="d-flex align-items-center mb-2">
                    <span class="status-badge status-approved me-2">Disetujui</span>
                    <small>Surat telah disetujui dan ditandatangani</small>
                </div>
                <div class="d-flex align-items-center">
                    <span class="status-badge status-rejected me-2">Ditolak</span>
                    <small>Surat ditolak dengan alasan tertentu</small>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
// File size validation
document.getElementById('document_file').addEventListener('change', function() {
    const file = this.files[0];
    const maxSize = <?php echo MAX_FILE_SIZE; ?>;
    
    if (file && file.size > maxSize) {
        alert('Ukuran file terlalu besar! Maksimal <?php echo formatFileSize(MAX_FILE_SIZE); ?>');
        this.value = '';
    }
});

// Auto-generate title from subject
document.getElementById('subject').addEventListener('input', function() {
    const titleField = document.getElementById('title');
    if (!titleField.value) {
        titleField.value = 'Surat ' + this.value;
    }
});
</script>

<?php include '../resources/footer.php'; ?>

