<?php
/**
 * Documents Management Page
 * Aplikasi TTD SMAN 1 Tanjungbalai
 * Developer: T. Rifaldo Sany
 */

require_once '../config/database.php';
require_once '../app/auth.php';

$auth = new Auth();
$auth->requireRole('administrator');

$db = getDB();
$error = '';
$success = '';

// Handle delete action
if (isset($_GET['delete']) && is_numeric($_GET['delete'])) {
    $documentId = $_GET['delete'];
    
    try {
        // Get document info first
        $stmt = $db->prepare("SELECT * FROM documents WHERE id = ? AND created_by = ?");
        $stmt->execute([$documentId, $_SESSION['user_id']]);
        $document = $stmt->fetch();
        
        if ($document) {
            // Delete file
            $filePath = DOCUMENT_PATH . '/' . $document['file_path'];
            if (file_exists($filePath)) {
                unlink($filePath);
            }
            
            // Delete QR code if exists
            if ($document['qr_code_path']) {
                $qrPath = QR_CODE_PATH . '/' . $document['qr_code_path'];
                if (file_exists($qrPath)) {
                    unlink($qrPath);
                }
            }
            
            // Delete from database
            $stmt = $db->prepare("DELETE FROM documents WHERE id = ? AND created_by = ?");
            if ($stmt->execute([$documentId, $_SESSION['user_id']])) {
                $success = 'Surat berhasil dihapus!';
            } else {
                $error = 'Gagal menghapus surat!';
            }
        } else {
            $error = 'Surat tidak ditemukan!';
        }
    } catch (PDOException $e) {
        $error = 'Terjadi kesalahan database!';
        error_log("Delete document error: " . $e->getMessage());
    }
}

// Get documents with pagination
$page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
$limit = 10;
$offset = ($page - 1) * $limit;

$search = isset($_GET['search']) ? sanitize($_GET['search']) : '';
$status = isset($_GET['status']) ? $_GET['status'] : '';

// Build query
$whereClause = "WHERE created_by = ?";
$params = [$_SESSION['user_id']];

if ($search) {
    $whereClause .= " AND (title LIKE ? OR document_number LIKE ? OR subject LIKE ?)";
    $searchParam = "%$search%";
    $params[] = $searchParam;
    $params[] = $searchParam;
    $params[] = $searchParam;
}

if ($status && in_array($status, ['pending', 'approved', 'rejected'])) {
    $whereClause .= " AND status = ?";
    $params[] = $status;
}

// Get total count
$countStmt = $db->prepare("SELECT COUNT(*) as total FROM documents $whereClause");
$countStmt->execute($params);
$totalDocuments = $countStmt->fetch()['total'];
$totalPages = ceil($totalDocuments / $limit);

// Get documents
$stmt = $db->prepare("SELECT * FROM documents $whereClause ORDER BY created_at DESC LIMIT $limit OFFSET $offset");
$stmt->execute($params);
$documents = $stmt->fetchAll();

$title = 'Kelola Surat - SMAN 1 Tanjungbalai';
$showNavbar = true;
$showSidebar = true;

include '../resources/header.php';
?>

<div class="d-flex justify-content-between align-items-center mb-4">
    <div>
        <h1 class="h3 mb-0 fw-bold">Kelola Surat</h1>
        <p class="text-muted mb-0">Kelola surat yang telah Anda buat</p>
    </div>
    <a href="create_document.php" class="btn btn-primary">
        <i class="fas fa-plus me-1"></i>Buat Surat Baru
    </a>
</div>

<?php if ($error): ?>
    <div class="alert alert-danger" role="alert">
        <i class="fas fa-exclamation-triangle me-2"></i><?php echo $error; ?>
    </div>
<?php endif; ?>

<?php if ($success): ?>
    <div class="alert alert-success" role="alert">
        <i class="fas fa-check-circle me-2"></i><?php echo $success; ?>
    </div>
<?php endif; ?>

<!-- Filters -->
<div class="card mb-4">
    <div class="card-body">
        <form method="GET" class="row g-3">
            <div class="col-md-4">
                <label for="search" class="form-label">Pencarian</label>
                <input type="text" class="form-control" id="search" name="search" 
                       placeholder="Cari judul, nomor, atau perihal..." 
                       value="<?php echo htmlspecialchars($search); ?>">
            </div>
            
            <div class="col-md-3">
                <label for="status" class="form-label">Status</label>
                <select class="form-select" id="status" name="status">
                    <option value="">Semua Status</option>
                    <option value="pending" <?php echo $status === 'pending' ? 'selected' : ''; ?>>Menunggu</option>
                    <option value="approved" <?php echo $status === 'approved' ? 'selected' : ''; ?>>Disetujui</option>
                    <option value="rejected" <?php echo $status === 'rejected' ? 'selected' : ''; ?>>Ditolak</option>
                </select>
            </div>
            
            <div class="col-md-3 d-flex align-items-end">
                <button type="submit" class="btn btn-outline-primary me-2">
                    <i class="fas fa-search me-1"></i>Cari
                </button>
                <a href="documents.php" class="btn btn-outline-secondary">
                    <i class="fas fa-undo me-1"></i>Reset
                </a>
            </div>
            
            <div class="col-md-2 d-flex align-items-end justify-content-end">
                <small class="text-muted">
                    Total: <?php echo $totalDocuments; ?> surat
                </small>
            </div>
        </form>
    </div>
</div>

<!-- Documents List -->
<div class="card">
    <div class="card-header bg-white">
        <h6 class="mb-0 fw-bold">Daftar Surat</h6>
    </div>
    <div class="card-body p-0">
        <?php if (empty($documents)): ?>
            <div class="text-center py-5">
                <i class="fas fa-inbox fa-3x text-muted mb-3"></i>
                <h6 class="text-muted">Belum ada surat</h6>
                <p class="text-muted mb-3">Mulai dengan membuat surat baru</p>
                <a href="create_document.php" class="btn btn-primary">
                    <i class="fas fa-plus me-1"></i>Buat Surat Baru
                </a>
            </div>
        <?php else: ?>
            <div class="table-responsive">
                <table class="table table-hover mb-0">
                    <thead class="table-light">
                        <tr>
                            <th>No. Surat</th>
                            <th>Judul</th>
                            <th>Perihal</th>
                            <th>Status</th>
                            <th>Tanggal</th>
                            <th>Aksi</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($documents as $doc): ?>
                            <tr>
                                <td>
                                    <span class="fw-semibold"><?php echo $doc['document_number']; ?></span>
                                </td>
                                <td>
                                    <div class="fw-semibold"><?php echo $doc['title']; ?></div>
                                    <?php if ($doc['description']): ?>
                                        <small class="text-muted"><?php echo substr($doc['description'], 0, 50) . (strlen($doc['description']) > 50 ? '...' : ''); ?></small>
                                    <?php endif; ?>
                                </td>
                                <td><?php echo $doc['subject']; ?></td>
                                <td>
                                    <span class="status-badge status-<?php echo $doc['status']; ?>">
                                        <?php 
                                        switch($doc['status']) {
                                            case 'pending': echo 'Menunggu'; break;
                                            case 'approved': echo 'Disetujui'; break;
                                            case 'rejected': echo 'Ditolak'; break;
                                        }
                                        ?>
                                    </span>
                                </td>
                                <td>
                                    <div><?php echo date('d/m/Y', strtotime($doc['created_at'])); ?></div>
                                    <small class="text-muted"><?php echo date('H:i', strtotime($doc['created_at'])); ?></small>
                                </td>
                                <td>
                                    <div class="btn-group btn-group-sm" role="group">
                                        <a href="view_document.php?id=<?php echo $doc['id']; ?>" 
                                           class="btn btn-outline-info" title="Lihat">
                                            <i class="fas fa-eye"></i>
                                        </a>
                                        
                                        <?php if ($doc['status'] === 'pending'): ?>
                                            <a href="edit_document.php?id=<?php echo $doc['id']; ?>" 
                                               class="btn btn-outline-warning" title="Edit">
                                                <i class="fas fa-edit"></i>
                                            </a>
                                        <?php endif; ?>
                                        
                                        <?php if ($doc['status'] === 'approved' && $doc['qr_code_path']): ?>
                                            <a href="../uploads/qr_codes/<?php echo $doc['qr_code_path']; ?>" 
                                               class="btn btn-outline-success" title="Download QR Code" download>
                                                <i class="fas fa-qrcode"></i>
                                            </a>
                                        <?php endif; ?>
                                        
                                        <a href="../uploads/documents/<?php echo $doc['file_path']; ?>" 
                                           class="btn btn-outline-primary" title="Download File" target="_blank">
                                            <i class="fas fa-download"></i>
                                        </a>
                                        
                                        <?php if ($doc['status'] === 'pending'): ?>
                                            <a href="?delete=<?php echo $doc['id']; ?>" 
                                               class="btn btn-outline-danger" title="Hapus"
                                               onclick="return confirm('Yakin ingin menghapus surat ini?')">
                                                <i class="fas fa-trash"></i>
                                            </a>
                                        <?php endif; ?>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
            
            <!-- Pagination -->
            <?php if ($totalPages > 1): ?>
                <div class="card-footer bg-white">
                    <nav aria-label="Page navigation">
                        <ul class="pagination pagination-sm justify-content-center mb-0">
                            <?php if ($page > 1): ?>
                                <li class="page-item">
                                    <a class="page-link" href="?page=<?php echo $page - 1; ?>&search=<?php echo urlencode($search); ?>&status=<?php echo $status; ?>">
                                        <i class="fas fa-chevron-left"></i>
                                    </a>
                                </li>
                            <?php endif; ?>
                            
                            <?php for ($i = max(1, $page - 2); $i <= min($totalPages, $page + 2); $i++): ?>
                                <li class="page-item <?php echo $i === $page ? 'active' : ''; ?>">
                                    <a class="page-link" href="?page=<?php echo $i; ?>&search=<?php echo urlencode($search); ?>&status=<?php echo $status; ?>">
                                        <?php echo $i; ?>
                                    </a>
                                </li>
                            <?php endfor; ?>
                            
                            <?php if ($page < $totalPages): ?>
                                <li class="page-item">
                                    <a class="page-link" href="?page=<?php echo $page + 1; ?>&search=<?php echo urlencode($search); ?>&status=<?php echo $status; ?>">
                                        <i class="fas fa-chevron-right"></i>
                                    </a>
                                </li>
                            <?php endif; ?>
                        </ul>
                    </nav>
                </div>
            <?php endif; ?>
        <?php endif; ?>
    </div>
</div>

<?php include '../resources/footer.php'; ?>

