<?php
/**
 * Edit Document Page
 * Aplikasi TTD SMAN 1 Tanjungbalai
 * Developer: T. Rifaldo Sany
 */

require_once '../config/database.php';
require_once '../app/auth.php';

$auth = new Auth();
$auth->requireRole('administrator');

$db = getDB();
$error = '';
$success = '';
$document = null;

if (!isset($_GET['id']) || !is_numeric($_GET['id'])) {
    $error = 'ID dokumen tidak valid!';
} else {
    $documentId = $_GET['id'];
    
    try {
        // Get document
        $stmt = $db->prepare("SELECT * FROM documents WHERE id = ? AND created_by = ? AND status = 'pending'");
        $stmt->execute([$documentId, $_SESSION['user_id']]);
        $document = $stmt->fetch();
        
        if (!$document) {
            $error = 'Dokumen tidak ditemukan atau tidak dapat diedit!';
        }
    } catch (PDOException $e) {
        $error = 'Terjadi kesalahan database!';
        error_log("Edit document error: " . $e->getMessage());
    }
}

if ($_SERVER['REQUEST_METHOD'] === 'POST' && $document) {
    $title = sanitize($_POST['title']);
    $subject = sanitize($_POST['subject']);
    $description = sanitize($_POST['description']);
    
    if (empty($title) || empty($subject)) {
        $error = 'Judul dan perihal surat harus diisi!';
    } else {
        try {
            $db->beginTransaction();
            
            // Update document info
            $stmt = $db->prepare("UPDATE documents SET title = ?, subject = ?, description = ?, updated_at = CURRENT_TIMESTAMP WHERE id = ? AND created_by = ?");
            $stmt->execute([$title, $subject, $description, $documentId, $_SESSION['user_id']]);
            
            // Handle file upload if new file is provided
            if (isset($_FILES['document_file']) && $_FILES['document_file']['error'] === UPLOAD_ERR_OK) {
                $file = $_FILES['document_file'];
                $allowedTypes = ['application/pdf', 'application/msword', 'application/vnd.openxmlformats-officedocument.wordprocessingml.document'];
                
                if (!in_array($file['type'], $allowedTypes)) {
                    throw new Exception('Format file tidak didukung! Gunakan PDF, DOC, atau DOCX.');
                } elseif ($file['size'] > MAX_FILE_SIZE) {
                    throw new Exception('Ukuran file terlalu besar! Maksimal ' . formatFileSize(MAX_FILE_SIZE) . '.');
                } else {
                    // Delete old file
                    $oldFilePath = DOCUMENT_PATH . '/' . $document['file_path'];
                    if (file_exists($oldFilePath)) {
                        unlink($oldFilePath);
                    }
                    
                    // Upload new file
                    $extension = pathinfo($file['name'], PATHINFO_EXTENSION);
                    $filename = $document['document_number'] . '_' . time() . '.' . $extension;
                    $uploadPath = DOCUMENT_PATH . '/' . $filename;
                    
                    if (move_uploaded_file($file['tmp_name'], $uploadPath)) {
                        // Update file info in database
                        $stmt = $db->prepare("UPDATE documents SET file_path = ?, file_name = ?, file_size = ? WHERE id = ?");
                        $stmt->execute([$filename, $file['name'], $file['size'], $documentId]);
                    } else {
                        throw new Exception('Gagal mengupload file!');
                    }
                }
            }
            
            // Log the edit action
            $logStmt = $db->prepare("INSERT INTO approval_logs (document_id, user_id, action, notes) VALUES (?, ?, 'edited', ?)");
            $logStmt->execute([$documentId, $_SESSION['user_id'], "Dokumen diedit: $title"]);
            
            $db->commit();
            $success = 'Dokumen berhasil diperbarui!';
            
            // Refresh document data
            $stmt = $db->prepare("SELECT * FROM documents WHERE id = ?");
            $stmt->execute([$documentId]);
            $document = $stmt->fetch();
            
        } catch (Exception $e) {
            $db->rollback();
            $error = $e->getMessage();
        } catch (PDOException $e) {
            $db->rollback();
            $error = 'Terjadi kesalahan database!';
            error_log("Edit document error: " . $e->getMessage());
        }
    }
}

$title = 'Edit Surat - SMAN 1 Tanjungbalai';
$showNavbar = true;
$showSidebar = true;

include '../resources/header.php';
?>

<div class="d-flex justify-content-between align-items-center mb-4">
    <div>
        <h1 class="h3 mb-0 fw-bold">Edit Surat</h1>
        <p class="text-muted mb-0">Edit informasi surat yang masih menunggu persetujuan</p>
    </div>
    <a href="documents.php" class="btn btn-outline-secondary">
        <i class="fas fa-arrow-left me-1"></i>Kembali
    </a>
</div>

<?php if ($error): ?>
    <div class="alert alert-danger" role="alert">
        <i class="fas fa-exclamation-triangle me-2"></i><?php echo $error; ?>
    </div>
    
    <?php if (!$document): ?>
        <div class="text-center">
            <a href="documents.php" class="btn btn-primary">
                <i class="fas fa-arrow-left me-1"></i>Kembali ke Daftar Surat
            </a>
        </div>
    <?php endif; ?>
<?php endif; ?>

<?php if ($success): ?>
    <div class="alert alert-success" role="alert">
        <i class="fas fa-check-circle me-2"></i><?php echo $success; ?>
        <div class="mt-2">
            <a href="view_document.php?id=<?php echo $documentId; ?>" class="btn btn-sm btn-success">Lihat Detail</a>
            <a href="documents.php" class="btn btn-sm btn-outline-success">Kembali ke Daftar</a>
        </div>
    </div>
<?php endif; ?>

<?php if ($document): ?>
<div class="row">
    <div class="col-lg-8">
        <div class="card">
            <div class="card-header bg-white">
                <div class="d-flex justify-content-between align-items-center">
                    <h6 class="mb-0 fw-bold">Edit Informasi Surat</h6>
                    <span class="badge bg-warning">Menunggu Persetujuan</span>
                </div>
            </div>
            <div class="card-body">
                <form method="POST" enctype="multipart/form-data" class="needs-validation" novalidate>
                    <div class="mb-3">
                        <label for="document_number" class="form-label fw-semibold">Nomor Surat</label>
                        <input type="text" class="form-control" id="document_number" 
                               value="<?php echo $document['document_number']; ?>" readonly>
                        <small class="text-muted">Nomor surat tidak dapat diubah</small>
                    </div>
                    
                    <div class="mb-3">
                        <label for="title" class="form-label fw-semibold">Judul Surat <span class="text-danger">*</span></label>
                        <input type="text" class="form-control" id="title" name="title" 
                               placeholder="Masukkan judul surat" 
                               value="<?php echo htmlspecialchars($document['title']); ?>" required>
                        <div class="invalid-feedback">
                            Judul surat harus diisi!
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="subject" class="form-label fw-semibold">Perihal <span class="text-danger">*</span></label>
                        <input type="text" class="form-control" id="subject" name="subject" 
                               placeholder="Masukkan perihal surat" 
                               value="<?php echo htmlspecialchars($document['subject']); ?>" required>
                        <div class="invalid-feedback">
                            Perihal surat harus diisi!
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="description" class="form-label fw-semibold">Deskripsi</label>
                        <textarea class="form-control" id="description" name="description" rows="4" 
                                  placeholder="Masukkan deskripsi atau keterangan tambahan"><?php echo htmlspecialchars($document['description']); ?></textarea>
                        <small class="text-muted">Opsional - Tambahkan keterangan atau catatan khusus</small>
                    </div>
                    
                    <div class="mb-4">
                        <label for="document_file" class="form-label fw-semibold">File Surat</label>
                        
                        <!-- Current File Info -->
                        <div class="alert alert-info">
                            <div class="d-flex align-items-center">
                                <i class="fas fa-file-pdf fa-2x text-danger me-3"></i>
                                <div class="flex-grow-1">
                                    <div class="fw-semibold">File Saat Ini: <?php echo $document['file_name']; ?></div>
                                    <small class="text-muted"><?php echo formatFileSize($document['file_size']); ?></small>
                                </div>
                                <div>
                                    <a href="../uploads/documents/<?php echo $document['file_path']; ?>" 
                                       class="btn btn-sm btn-outline-primary" target="_blank">
                                        <i class="fas fa-eye me-1"></i>Lihat
                                    </a>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Upload New File -->
                        <input type="file" class="form-control" id="document_file" name="document_file" 
                               accept=".pdf,.doc,.docx">
                        <small class="text-muted">
                            Opsional - Upload file baru jika ingin mengganti. Format: PDF, DOC, DOCX. Maksimal <?php echo formatFileSize(MAX_FILE_SIZE); ?>
                        </small>
                    </div>
                    
                    <div class="d-flex gap-2">
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-save me-1"></i>Simpan Perubahan
                        </button>
                        <a href="view_document.php?id=<?php echo $documentId; ?>" class="btn btn-outline-info">
                            <i class="fas fa-eye me-1"></i>Lihat Detail
                        </a>
                        <a href="documents.php" class="btn btn-outline-secondary">
                            <i class="fas fa-times me-1"></i>Batal
                        </a>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <div class="col-lg-4">
        <!-- Document Info -->
        <div class="card mb-4">
            <div class="card-header bg-primary text-white">
                <h6 class="mb-0 fw-bold">
                    <i class="fas fa-info-circle me-2"></i>Informasi Dokumen
                </h6>
            </div>
            <div class="card-body">
                <div class="mb-3">
                    <small class="text-muted">Status</small>
                    <div><span class="status-badge status-pending">Menunggu Persetujuan</span></div>
                </div>
                
                <div class="mb-3">
                    <small class="text-muted">Dibuat</small>
                    <div><?php echo date('d F Y, H:i', strtotime($document['created_at'])); ?></div>
                </div>
                
                <div class="mb-3">
                    <small class="text-muted">Terakhir Diubah</small>
                    <div><?php echo date('d F Y, H:i', strtotime($document['updated_at'])); ?></div>
                </div>
                
                <div>
                    <small class="text-muted">Ukuran File</small>
                    <div><?php echo formatFileSize($document['file_size']); ?></div>
                </div>
            </div>
        </div>
        
        <!-- Edit Guidelines -->
        <div class="card">
            <div class="card-header bg-white">
                <h6 class="mb-0 fw-bold">Panduan Edit</h6>
            </div>
            <div class="card-body">
                <div class="mb-3">
                    <h6 class="fw-semibold">Yang Dapat Diedit:</h6>
                    <ul class="small mb-0">
                        <li>Judul surat</li>
                        <li>Perihal surat</li>
                        <li>Deskripsi/keterangan</li>
                        <li>File surat (opsional)</li>
                    </ul>
                </div>
                
                <div class="mb-3">
                    <h6 class="fw-semibold">Catatan Penting:</h6>
                    <ul class="small mb-0">
                        <li>Hanya surat dengan status "Menunggu" yang dapat diedit</li>
                        <li>Nomor surat tidak dapat diubah</li>
                        <li>Setelah disetujui/ditolak, surat tidak dapat diedit</li>
                        <li>File baru akan mengganti file lama</li>
                    </ul>
                </div>
                
                <div>
                    <h6 class="fw-semibold">Tips:</h6>
                    <ul class="small mb-0">
                        <li>Periksa kembali informasi sebelum menyimpan</li>
                        <li>Pastikan file dapat dibaca dengan jelas</li>
                        <li>Gunakan format PDF untuk hasil terbaik</li>
                    </ul>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
// File size validation
document.getElementById('document_file').addEventListener('change', function() {
    const file = this.files[0];
    const maxSize = <?php echo MAX_FILE_SIZE; ?>;
    
    if (file && file.size > maxSize) {
        alert('Ukuran file terlalu besar! Maksimal <?php echo formatFileSize(MAX_FILE_SIZE); ?>');
        this.value = '';
    }
});
</script>

<?php endif; ?>

<?php include '../resources/footer.php'; ?>

