<?php
/**
 * Document Verification Page (Public)
 * Aplikasi TTD SMAN 1 Tanjungbalai
 * Developer: T. Rifaldo Sany
 */

require_once '../config/database.php';

$db = getDB();
$document = null;
$error = '';
$documentNumber = '';

if (isset($_GET['doc'])) {
    $documentNumber = sanitize($_GET['doc']);
    
    try {
        // Get document with creator and approver info
        $stmt = $db->prepare("SELECT d.*, 
                             creator.full_name as creator_name, creator.position as creator_position,
                             approver.full_name as approver_name, approver.position as approver_position
                             FROM documents d 
                             JOIN users creator ON d.created_by = creator.id 
                             LEFT JOIN users approver ON d.approved_by = approver.id
                             WHERE d.document_number = ?");
        $stmt->execute([$documentNumber]);
        $document = $stmt->fetch();
        
        if (!$document) {
            $error = 'Dokumen tidak ditemukan!';
        }
        
    } catch (PDOException $e) {
        $error = 'Terjadi kesalahan sistem!';
        error_log("Verification error: " . $e->getMessage());
    }
}

$title = 'Verifikasi Surat - SMAN 1 Tanjungbalai';
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $title; ?></title>
    
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <!-- Font Awesome -->
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" rel="stylesheet">
    
    <style>
        :root {
            --primary-color: #2E7D32;
            --secondary-color: #4CAF50;
            --accent-color: #FFC107;
        }
        
        body {
            background: linear-gradient(135deg, #E8F5E8 0%, #ffffff 50%, #E8F5E8 100%);
            min-height: 100vh;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }
        
        .verification-container {
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px 0;
        }
        
        .verification-card {
            background: white;
            border-radius: 20px;
            box-shadow: 0 20px 40px rgba(0,0,0,0.1);
            overflow: hidden;
            max-width: 800px;
            width: 100%;
        }
        
        .card-header-custom {
            background: linear-gradient(135deg, var(--primary-color), var(--secondary-color));
            color: white;
            padding: 30px;
            text-align: center;
        }
        
        .status-badge {
            padding: 8px 16px;
            border-radius: 20px;
            font-size: 0.85em;
            font-weight: 600;
        }
        
        .status-pending {
            background: #FFF3CD;
            color: #856404;
            animation: pulse 2s infinite;
        }
        
        .status-approved {
            background: #D4EDDA;
            color: #155724;
        }
        
        .status-rejected {
            background: #F8D7DA;
            color: #721C24;
        }
        
        @keyframes pulse {
            0% { opacity: 1; }
            50% { opacity: 0.7; }
            100% { opacity: 1; }
        }
        
        .verification-icon {
            width: 80px;
            height: 80px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0 auto 20px;
            font-size: 2rem;
        }
        
        .icon-approved {
            background: #D4EDDA;
            color: #155724;
        }
        
        .icon-rejected {
            background: #F8D7DA;
            color: #721C24;
        }
        
        .icon-pending {
            background: #FFF3CD;
            color: #856404;
        }
        
        .icon-error {
            background: #F8D7DA;
            color: #721C24;
        }
        
        .search-form {
            background: white;
            border-radius: 15px;
            padding: 30px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.1);
        }
    </style>
</head>
<body>
    <div class="verification-container">
        <div class="container">
            <div class="row justify-content-center">
                <div class="col-lg-10">
                    <?php if (!$documentNumber): ?>
                        <!-- Search Form -->
                        <div class="search-form text-center">
                            <div class="mb-4">
                                <img src="../uploads/logo_pemprovsu.png" alt="Logo Pemprovsu" style="height: 80px;" class="mb-3">
                                <h2 class="fw-bold text-dark">Verifikasi Surat Digital</h2>
                                <p class="text-muted">SMAN 1 Tanjungbalai</p>
                            </div>
                            
                            <form method="GET" class="row g-3 justify-content-center">
                                <div class="col-md-8">
                                    <label for="doc" class="form-label fw-semibold">Nomor Surat</label>
                                    <input type="text" class="form-control form-control-lg" id="doc" name="doc" 
                                           placeholder="Masukkan nomor surat (contoh: 001/SMAN-01/01/2025)" required>
                                </div>
                                <div class="col-md-4 d-flex align-items-end">
                                    <button type="submit" class="btn btn-primary btn-lg w-100">
                                        <i class="fas fa-search me-2"></i>Verifikasi
                                    </button>
                                </div>
                            </form>
                            
                            <div class="mt-4">
                                <small class="text-muted">
                                    Masukkan nomor surat untuk memverifikasi keaslian dokumen digital SMAN 1 Tanjungbalai
                                </small>
                            </div>
                        </div>
                    
                    <?php else: ?>
                        <!-- Verification Result -->
                        <div class="verification-card">
                            <div class="card-header-custom">
                                <img src="../uploads/logo_pemprovsu.png" alt="Logo Pemprovsu" style="height: 60px;" class="mb-3">
                                <h3 class="fw-bold mb-0">Hasil Verifikasi Surat</h3>
                                <p class="mb-0 opacity-75">SMAN 1 Tanjungbalai</p>
                            </div>
                            
                            <div class="card-body p-4">
                                <?php if ($error): ?>
                                    <!-- Error State -->
                                    <div class="text-center mb-4">
                                        <div class="verification-icon icon-error">
                                            <i class="fas fa-exclamation-triangle"></i>
                                        </div>
                                        <h4 class="fw-bold text-danger">Dokumen Tidak Ditemukan</h4>
                                        <p class="text-muted"><?php echo $error; ?></p>
                                        <p class="small text-muted">Nomor surat: <strong><?php echo htmlspecialchars($documentNumber); ?></strong></p>
                                    </div>
                                    
                                    <div class="alert alert-warning">
                                        <h6 class="fw-bold">Kemungkinan Penyebab:</h6>
                                        <ul class="mb-0">
                                            <li>Nomor surat salah atau tidak valid</li>
                                            <li>Dokumen belum disetujui atau masih dalam proses</li>
                                            <li>Dokumen telah dihapus dari sistem</li>
                                        </ul>
                                    </div>
                                
                                <?php else: ?>
                                    <!-- Document Found -->
                                    <div class="text-center mb-4">
                                        <?php if ($document['status'] === 'approved'): ?>
                                            <div class="verification-icon icon-approved">
                                                <i class="fas fa-check-circle"></i>
                                            </div>
                                            <h4 class="fw-bold text-success">Dokumen Valid & Terverifikasi</h4>
                                            <p class="text-muted">Dokumen ini telah disetujui dan ditandatangani secara digital</p>
                                        
                                        <?php elseif ($document['status'] === 'pending'): ?>
                                            <div class="verification-icon icon-pending">
                                                <i class="fas fa-clock"></i>
                                            </div>
                                            <h4 class="fw-bold text-warning">Dokumen Menunggu Persetujuan</h4>
                                            <p class="text-muted">Dokumen masih dalam proses persetujuan</p>
                                        
                                        <?php else: ?>
                                            <div class="verification-icon icon-rejected">
                                                <i class="fas fa-times-circle"></i>
                                            </div>
                                            <h4 class="fw-bold text-danger">Dokumen Ditolak</h4>
                                            <p class="text-muted">Dokumen telah ditolak oleh pihak berwenang</p>
                                        <?php endif; ?>
                                    </div>
                                    
                                    <!-- Document Details -->
                                    <div class="row">
                                        <div class="col-md-6 mb-3">
                                            <label class="form-label fw-semibold text-muted small">NOMOR SURAT</label>
                                            <p class="mb-0 fw-bold"><?php echo $document['document_number']; ?></p>
                                        </div>
                                        
                                        <div class="col-md-6 mb-3">
                                            <label class="form-label fw-semibold text-muted small">STATUS</label>
                                            <div>
                                                <span class="status-badge status-<?php echo $document['status']; ?>">
                                                    <?php 
                                                    switch($document['status']) {
                                                        case 'pending': echo 'Menunggu Persetujuan'; break;
                                                        case 'approved': echo 'Disetujui & Ditandatangani'; break;
                                                        case 'rejected': echo 'Ditolak'; break;
                                                    }
                                                    ?>
                                                </span>
                                            </div>
                                        </div>
                                        
                                        <div class="col-12 mb-3">
                                            <label class="form-label fw-semibold text-muted small">JUDUL SURAT</label>
                                            <p class="mb-0"><?php echo $document['title']; ?></p>
                                        </div>
                                        
                                        <div class="col-12 mb-3">
                                            <label class="form-label fw-semibold text-muted small">PERIHAL</label>
                                            <p class="mb-0"><?php echo $document['subject']; ?></p>
                                        </div>
                                        
                                        <div class="col-md-6 mb-3">
                                            <label class="form-label fw-semibold text-muted small">DIBUAT OLEH</label>
                                            <p class="mb-0">
                                                <?php echo $document['creator_name']; ?><br>
                                                <small class="text-muted"><?php echo $document['creator_position']; ?></small>
                                            </p>
                                        </div>
                                        
                                        <div class="col-md-6 mb-3">
                                            <label class="form-label fw-semibold text-muted small">TANGGAL DIBUAT</label>
                                            <p class="mb-0"><?php echo date('d F Y', strtotime($document['created_at'])); ?></p>
                                        </div>
                                        
                                        <?php if ($document['approved_by']): ?>
                                        <div class="col-md-6 mb-3">
                                            <label class="form-label fw-semibold text-muted small">
                                                <?php echo $document['status'] === 'approved' ? 'DISETUJUI OLEH' : 'DITOLAK OLEH'; ?>
                                            </label>
                                            <p class="mb-0">
                                                <?php echo $document['approver_name']; ?><br>
                                                <small class="text-muted"><?php echo $document['approver_position']; ?></small>
                                            </p>
                                        </div>
                                        
                                        <div class="col-md-6 mb-3">
                                            <label class="form-label fw-semibold text-muted small">TANGGAL PERSETUJUAN</label>
                                            <p class="mb-0"><?php echo date('d F Y', strtotime($document['approved_at'])); ?></p>
                                        </div>
                                        <?php endif; ?>
                                        
                                        <?php if ($document['status'] === 'rejected' && $document['rejection_reason']): ?>
                                        <div class="col-12 mb-3">
                                            <label class="form-label fw-semibold text-muted small">ALASAN PENOLAKAN</label>
                                            <div class="alert alert-danger">
                                                <?php echo nl2br($document['rejection_reason']); ?>
                                            </div>
                                        </div>
                                        <?php endif; ?>
                                    </div>
                                    
                                    <!-- Verification Info -->
                                    <div class="border-top pt-3 mt-3">
                                        <div class="row align-items-center">
                                            <div class="col-md-8">
                                                <h6 class="fw-semibold mb-1">Informasi Verifikasi</h6>
                                                <p class="small text-muted mb-0">
                                                    Dokumen ini telah diverifikasi melalui sistem digital SMAN 1 Tanjungbalai. 
                                                    Waktu verifikasi: <?php echo date('d F Y, H:i:s'); ?>
                                                </p>
                                            </div>
                                            <div class="col-md-4 text-md-end">
                                                <i class="fas fa-shield-alt fa-2x text-success"></i>
                                            </div>
                                        </div>
                                    </div>
                                <?php endif; ?>
                                
                                <!-- Actions -->
                                <div class="text-center mt-4">
                                    <a href="verify.php" class="btn btn-outline-primary me-2">
                                        <i class="fas fa-search me-1"></i>Verifikasi Lain
                                    </a>
                                    
                                    <?php if (!$error && $document['status'] === 'approved'): ?>
                                        <button onclick="window.print()" class="btn btn-outline-secondary">
                                            <i class="fas fa-print me-1"></i>Cetak Hasil
                                        </button>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Bootstrap JS -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>

