<?php
/**
 * View Document Page
 * Aplikasi TTD SMAN 1 Tanjungbalai
 * Developer: T. Rifaldo Sany
 */

require_once '../config/database.php';
require_once '../app/auth.php';

$auth = new Auth();
$auth->requireLogin();

$db = getDB();
$error = '';
$document = null;

if (!isset($_GET['id']) || !is_numeric($_GET['id'])) {
    $error = 'ID surat tidak valid!';
} else {
    $documentId = $_GET['id'];
    
    try {
        // Get document with creator info
        $stmt = $db->prepare("SELECT d.*, u.full_name as creator_name, u.position as creator_position,
                             approver.full_name as approver_name, approver.position as approver_position
                             FROM documents d 
                             JOIN users u ON d.created_by = u.id 
                             LEFT JOIN users approver ON d.approved_by = approver.id
                             WHERE d.id = ?");
        $stmt->execute([$documentId]);
        $document = $stmt->fetch();
        
        if (!$document) {
            $error = 'Surat tidak ditemukan!';
        } elseif ($_SESSION['role'] === 'administrator' && $document['created_by'] != $_SESSION['user_id']) {
            $error = 'Anda tidak memiliki akses ke surat ini!';
        }
        
        // Get approval logs
        $logStmt = $db->prepare("SELECT al.*, u.full_name as user_name 
                               FROM approval_logs al 
                               JOIN users u ON al.user_id = u.id 
                               WHERE al.document_id = ? 
                               ORDER BY al.created_at DESC");
        $logStmt->execute([$documentId]);
        $logs = $logStmt->fetchAll();
        
    } catch (PDOException $e) {
        $error = 'Terjadi kesalahan database!';
        error_log("View document error: " . $e->getMessage());
    }
}

$title = 'Detail Surat - SMAN 1 Tanjungbalai';
$showNavbar = true;
$showSidebar = true;

include '../resources/header.php';
?>

<?php if ($error): ?>
    <div class="alert alert-danger" role="alert">
        <i class="fas fa-exclamation-triangle me-2"></i><?php echo $error; ?>
    </div>
    <div class="text-center">
        <a href="<?php echo $_SESSION['role'] === 'administrator' ? 'documents.php' : 'approvals.php'; ?>" class="btn btn-primary">
            <i class="fas fa-arrow-left me-1"></i>Kembali
        </a>
    </div>
<?php else: ?>

<div class="d-flex justify-content-between align-items-center mb-4">
    <div>
        <h1 class="h3 mb-0 fw-bold">Detail Surat</h1>
        <p class="text-muted mb-0"><?php echo $document['document_number']; ?></p>
    </div>
    <div>
        <a href="<?php echo $_SESSION['role'] === 'administrator' ? 'documents.php' : 'approvals.php'; ?>" class="btn btn-outline-secondary me-2">
            <i class="fas fa-arrow-left me-1"></i>Kembali
        </a>
        
        <?php if ($_SESSION['role'] === 'administrator' && $document['status'] === 'pending'): ?>
            <a href="edit_document.php?id=<?php echo $document['id']; ?>" class="btn btn-warning">
                <i class="fas fa-edit me-1"></i>Edit
            </a>
        <?php endif; ?>
    </div>
</div>

<div class="row">
    <!-- Document Information -->
    <div class="col-lg-8 mb-4">
        <div class="card">
            <div class="card-header bg-white">
                <div class="d-flex justify-content-between align-items-center">
                    <h6 class="mb-0 fw-bold">Informasi Surat</h6>
                    <span class="status-badge status-<?php echo $document['status']; ?>">
                        <?php 
                        switch($document['status']) {
                            case 'pending': echo 'Menunggu Persetujuan'; break;
                            case 'approved': echo 'Disetujui'; break;
                            case 'rejected': echo 'Ditolak'; break;
                        }
                        ?>
                    </span>
                </div>
            </div>
            <div class="card-body">
                <div class="row">
                    <div class="col-md-6 mb-3">
                        <label class="form-label fw-semibold text-muted">Nomor Surat</label>
                        <p class="mb-0"><?php echo $document['document_number']; ?></p>
                    </div>
                    
                    <div class="col-md-6 mb-3">
                        <label class="form-label fw-semibold text-muted">Tanggal Dibuat</label>
                        <p class="mb-0"><?php echo date('d F Y, H:i', strtotime($document['created_at'])); ?></p>
                    </div>
                    
                    <div class="col-12 mb-3">
                        <label class="form-label fw-semibold text-muted">Judul Surat</label>
                        <p class="mb-0"><?php echo $document['title']; ?></p>
                    </div>
                    
                    <div class="col-12 mb-3">
                        <label class="form-label fw-semibold text-muted">Perihal</label>
                        <p class="mb-0"><?php echo $document['subject']; ?></p>
                    </div>
                    
                    <?php if ($document['description']): ?>
                    <div class="col-12 mb-3">
                        <label class="form-label fw-semibold text-muted">Deskripsi</label>
                        <p class="mb-0"><?php echo nl2br($document['description']); ?></p>
                    </div>
                    <?php endif; ?>
                    
                    <div class="col-md-6 mb-3">
                        <label class="form-label fw-semibold text-muted">Dibuat Oleh</label>
                        <p class="mb-0">
                            <?php echo $document['creator_name']; ?><br>
                            <small class="text-muted"><?php echo $document['creator_position']; ?></small>
                        </p>
                    </div>
                    
                    <?php if ($document['approved_by']): ?>
                    <div class="col-md-6 mb-3">
                        <label class="form-label fw-semibold text-muted">
                            <?php echo $document['status'] === 'approved' ? 'Disetujui Oleh' : 'Ditolak Oleh'; ?>
                        </label>
                        <p class="mb-0">
                            <?php echo $document['approver_name']; ?><br>
                            <small class="text-muted"><?php echo $document['approver_position']; ?></small><br>
                            <small class="text-muted"><?php echo date('d F Y, H:i', strtotime($document['approved_at'])); ?></small>
                        </p>
                    </div>
                    <?php endif; ?>
                    
                    <?php if ($document['status'] === 'rejected' && $document['rejection_reason']): ?>
                    <div class="col-12 mb-3">
                        <label class="form-label fw-semibold text-muted">Alasan Penolakan</label>
                        <div class="alert alert-danger">
                            <?php echo nl2br($document['rejection_reason']); ?>
                        </div>
                    </div>
                    <?php endif; ?>
                </div>
                
                <!-- File Actions -->
                <div class="border-top pt-3">
                    <h6 class="fw-semibold mb-3">File Surat</h6>
                    <div class="d-flex align-items-center">
                        <i class="fas fa-file-pdf fa-2x text-danger me-3"></i>
                        <div class="flex-grow-1">
                            <div class="fw-semibold"><?php echo $document['file_name']; ?></div>
                            <small class="text-muted"><?php echo formatFileSize($document['file_size']); ?></small>
                        </div>
                        <div>
                            <a href="../uploads/documents/<?php echo $document['file_path']; ?>" 
                               class="btn btn-outline-primary btn-sm me-2" target="_blank">
                                <i class="fas fa-eye me-1"></i>Lihat
                            </a>
                            <a href="../uploads/documents/<?php echo $document['file_path']; ?>" 
                               class="btn btn-primary btn-sm" download>
                                <i class="fas fa-download me-1"></i>Download
                            </a>
                        </div>
                    </div>
                </div>
                
                <!-- QR Code -->
                <?php if ($document['status'] === 'approved' && $document['qr_code_path']): ?>
                <div class="border-top pt-3 mt-3">
                    <h6 class="fw-semibold mb-3">QR Code Verifikasi</h6>
                    <div class="d-flex align-items-center">
                        <img src="../uploads/qr_codes/<?php echo $document['qr_code_path']; ?>" 
                             alt="QR Code" class="me-3" style="width: 100px; height: 100px;">
                        <div class="flex-grow-1">
                            <p class="mb-2">Scan QR Code untuk verifikasi keaslian surat</p>
                            <a href="../uploads/qr_codes/<?php echo $document['qr_code_path']; ?>" 
                               class="btn btn-success btn-sm" download>
                                <i class="fas fa-download me-1"></i>Download QR Code
                            </a>
                        </div>
                    </div>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
    
    <!-- Approval History -->
    <div class="col-lg-4">
        <div class="card">
            <div class="card-header bg-white">
                <h6 class="mb-0 fw-bold">Riwayat Persetujuan</h6>
            </div>
            <div class="card-body">
                <?php if (empty($logs)): ?>
                    <p class="text-muted text-center">Belum ada riwayat</p>
                <?php else: ?>
                    <div class="timeline">
                        <?php foreach ($logs as $log): ?>
                            <div class="timeline-item mb-3">
                                <div class="d-flex">
                                    <div class="timeline-marker me-3">
                                        <?php
                                        switch($log['action']) {
                                            case 'submitted':
                                                echo '<i class="fas fa-paper-plane text-primary"></i>';
                                                break;
                                            case 'approved':
                                                echo '<i class="fas fa-check-circle text-success"></i>';
                                                break;
                                            case 'rejected':
                                                echo '<i class="fas fa-times-circle text-danger"></i>';
                                                break;
                                            case 'edited':
                                                echo '<i class="fas fa-edit text-warning"></i>';
                                                break;
                                        }
                                        ?>
                                    </div>
                                    <div class="flex-grow-1">
                                        <div class="fw-semibold">
                                            <?php
                                            switch($log['action']) {
                                                case 'submitted':
                                                    echo 'Surat Dibuat';
                                                    break;
                                                case 'approved':
                                                    echo 'Surat Disetujui';
                                                    break;
                                                case 'rejected':
                                                    echo 'Surat Ditolak';
                                                    break;
                                                case 'edited':
                                                    echo 'Surat Diedit';
                                                    break;
                                            }
                                            ?>
                                        </div>
                                        <small class="text-muted">
                                            oleh <?php echo $log['user_name']; ?><br>
                                            <?php echo date('d/m/Y H:i', strtotime($log['created_at'])); ?>
                                        </small>
                                        <?php if ($log['notes']): ?>
                                            <div class="mt-1">
                                                <small><?php echo $log['notes']; ?></small>
                                            </div>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>
            </div>
        </div>
        
        <!-- Verification Link -->
        <?php if ($document['status'] === 'approved'): ?>
        <div class="card mt-3">
            <div class="card-header bg-white">
                <h6 class="mb-0 fw-bold">Link Verifikasi</h6>
            </div>
            <div class="card-body">
                <p class="small text-muted mb-2">Link untuk verifikasi publik:</p>
                <div class="input-group input-group-sm">
                    <input type="text" class="form-control" 
                           value="<?php echo APP_URL; ?>/verify.php?doc=<?php echo $document['document_number']; ?>" 
                           id="verificationLink" readonly>
                    <button class="btn btn-outline-secondary" type="button" onclick="copyLink()">
                        <i class="fas fa-copy"></i>
                    </button>
                </div>
            </div>
        </div>
        <?php endif; ?>
    </div>
</div>

<script>
function copyLink() {
    const linkInput = document.getElementById('verificationLink');
    linkInput.select();
    linkInput.setSelectionRange(0, 99999);
    document.execCommand('copy');
    
    // Show feedback
    const button = event.target.closest('button');
    const originalHTML = button.innerHTML;
    button.innerHTML = '<i class="fas fa-check"></i>';
    button.classList.remove('btn-outline-secondary');
    button.classList.add('btn-success');
    
    setTimeout(() => {
        button.innerHTML = originalHTML;
        button.classList.remove('btn-success');
        button.classList.add('btn-outline-secondary');
    }, 2000);
}
</script>

<?php endif; ?>

<?php include '../resources/footer.php'; ?>

